	
;------------------------------------------------------------------------------
; This file defines various constant parameter values, variable names (ram), 
; and macros. 
;------------------------------------------------------------------------------
;  PORTA BIT definitions 
;
;	PORTA<0> : IMOT, analog input for motor current
;	PORTA<1> : POT, analog input for speed control potentiometer
;	PORTA<2> : Hall A sensor input (IC1)
;	PORTA<3> : Hall B sensor input (IC2)
;	PORTA<4> : Hall C sensor input (IC3)
;	PORTA<5> : not present in this microcontroller version.
;	PORTA<6> : not present in this microcontroller version.
;	PORTA<7> : not present in this microcontroller version.
;------------------------------------------------------------------------------
#define TrisaValue	 		b'00011100' ; IC1/2/3 are inputs
;-------------------------------------------------------------------------------
;  PORTC BIT definitions 
;
;	PORTC<0> : push button mux enable
;	PORTC<1> : /FLTA interrupt input
;	PORTC<2> : Dip Switch, SW1, enable
;	PORTC<3> : LED enable (active low)
;	PORTC<4> : mux i/o (Start/Stop input, SW1-1 input, LED0 output)
;	PORTC<5> : mux i/o (Reverse    input, SW1-2 input, LED1 output)
;	PORTC<6> : mux i/o (Brake      input, SW1-3 input, LED2 output)
;	PORTC<7> : mux i/o (not used   input, SW1-4 input, LED3 output)
;-------------------------------------------------------------------------------
;  PORTB BIT definitions 
;
;	PORTB<0> : PWM0
;	PORTB<1> : PWM1
;	PORTB<2> : PWM2
;	PORTB<3> : PWM3
;	PORTB<4> : PWM5
;	PORTB<5> : PWM4
;	PORTB<6> : Output for ICD2
;	PORTB<7> : Input for ICD2
;-------------------------------------------------------------------------------
#define TrisbValue	 		b'10000000' 

;-------------------------------------------------------------------------------
;  VARIABLE (ram) DEFINITIONS 
;-------------------------------------------------------------------------------
BLDC_MOTOR_CONTROL			UDATA_ACS	;access ram

PWMCON0buf					res 1
OVDCONDbuf					res 1
PDCbuf_H					res 1
PDCbuf_L					res 1
STATUSsave					res 1
TBLPTRLsave					res 1

;TableSelectOffset			res 1	;Selects which switching sequence table to use.
;TableCounter				res 1	;Number of elements in the table.
;SequenceTable
;#define	TableSize		0x10	;SequenceFWD - SequenceREV
;-------------------------------------------------------------------------------
; ADC speed reference as set by potentiometer
;-------------------------------------------------------------------------------
SpeedRef_H					res	1	 
SpeedRef_L					res	1
PWMsetting_H				res 1
PWMsetting_L				res 1
LimitPWM_H					res 1
LimitPWM_L					res 1
PWMdutyCycle_H				res 1
PWMdutyCycle_L				res 1

;-------------------------------------------------------------------------------
; ADC measured motor current
;-------------------------------------------------------------------------------
MotorCurrent_H				res	1
MotorCurrent_L				res	1

;-------------------------------------------------------------------------------
;Multiplexing Macros
;  LEDs, push buttons, and dip switches are muxed sharing the RC4..RC7 i/0 pins.
;  The RC4..RC7 pins are configured as inputs for the push buttons and dips
;  switches, and are configured as outputs to drive the LEDs. 
;
;  RC0 is the mux select output pin for the push buttons, RC2 is the mux for
;  the dip switches, and RC3 is the select pin (active low) for the LEDs.
;  Note that RC1 is the microcontroller designated pin for the /FTA input
;  and is always configured as an input.
;
;  The default state for the muxed pins is for driving the LEDs. When muxing
;  the switches, the time to read the switches is kept to a minimum so as to
;  prevent the LEDs from flickering. After the switch data is acquired, the mux
;  state is always restored to the LED state.  
; 
;-------------------------------------------------------------------------------
LEDstatus			res 1
#define PBmuxPin 	0
#define SwitchMux	b'11110010' ;changes RC4..RC7 to inputs (b1 always an input)
#define LEDmux		b'00000010' ;changes RC4..RC7 to outputs(b1 always an input)
#define PushButton	0
#define DipSwitch	2
#define LEDmuxPin	3

EnableMux macro MuxPin, DataRegister
			bcf	  Timer0IntEnable_b
			movff LATC,LEDstatus	;save present state of the LEDs
			movlw b'00001000'		;Disable LEDs
			movwf LATC 				;  and discharge parasitic cap voltage
			movlw SwitchMux
			movwf TRISC          	;change RC4,5,6, and 7 to inputs
			bsf   LATC,MuxPin 		;turn on mux device as specified 
			movff PORTC,DataRegister;get switch state
			movff LEDstatus,LATC	;restore present state of the LEDs
			movlw LEDmux  
			movwf TRISC				;change RC4..RC7 to outputs
			bsf	  Timer0IntEnable_b
		  endm
				
;-------------------------------------------------------------------------------
;TurnOnLED and TurnOffLED macros
; The LEDnumber argument chooses which LED to turn on (or off) and the Blink
; argument enables/disables the blinking of that LED. Note that only the LEDs
; addressed by the LEDnumber are modified. All other LEDs remain on or off, and
; blinking or not blinking.
;
; For example:
;	TurnOnLED Brake,NoBlink -> the brake LED is turned on without blinking
;	TurnOnLED 0xF0, Blink   -> all LEDs (b4..b7) are turned on with blinking
;	TurnOnLED 0x90, Blink   -> LED0(b4) and LED3(B7) are turned on with blinking
; 
;	TurnOffLED Reverse		-> Reverse (LED1) is turned off
;	TurnoffLED 0x60			-> LED1 (b5) and LED2 (b6) are turned off
;
; The blinking of the LEDs is controlled by the TIMER0_INT routine. 
;-------------------------------------------------------------------------------

#define	Run				4
#define	LED0			4
#define StartStop		4
#define	Reverse			5
#define	LED1			5
#define	Brake			6
#define	LED2			6
#define	Ilimit			7
#define	LED3			7
#define Blink			1
#define NoBlink			0
#define Wdata			2

TurnOnLED	macro	LEDnumber, BlinkMode
			if 	(LEDnumber==4)|(LEDnumber==5)|(LEDnumber==6)|(LEDnumber==7)
				btfss EnableLEDs_B,LEDnumber
					bsf	LATC,LEDnumber;turn on specified LED only if not enabled
				bsf	EnableLEDs_B,LEDnumber ;enable specified LED for Timer0_Int
				if BlinkMode == 0
					bcf EnableBlink_B,LEDnumber 
				else
					bsf EnableBlink_B,LEDnumber
				endif
			else
				movwf LATC
				movwf EnableLEDs_B
				if BlinkMode == 1
					movwf EnableBlink_B
				else
					clrf EnableBlink_B
				endif	
			endif
			endm

TurnOffLED	macro LEDnumber
			if 	(LEDnumber==4)|(LEDnumber==5)|(LEDnumber==6)|(LEDnumber==7)
				bcf	LATC,LEDnumber
				bcf	EnableLEDs_B,LEDnumber
				bcf	EnableBlink_B,LEDnumber
			else
				movwf LATC
				movwf EnableLEDs_B
				andwf EnableBlink_B,W
				;movwf EnableBlink_B
			endif			
		  	endm
		  	
;-------------------------------------------------------------------------------
; Push-button and dip switch related parameters
;-------------------------------------------------------------------------------
DebounceCntr 				res 1
PrevSwInputs 				res 1
DipTest						res 1
PButtonInputs				res 1
DipSwInputs					res 1
;DebouncedSw				res 1

				
#define	PButtonMask			b'11110000'
#define DebounceCnt			0xFF

StartStop_B					res 1
Reverse_B					res 1
Brake_B						res 1
SpecialTestMode_B			res 1
#define SpecialTestMode_b	SpecialTestMode_B,0
#define	StartStop_b			StartStop_B,0	
#define	Reverse_b			Reverse_B,0	
#define	Brake_b				Brake_B,0


;-------------------------------------------------------------------------------
; Counters used for delays  
;-------------------------------------------------------------------------------
DelayCntr_H					res 1
DelayCntr_L					res 1
;UpdateDelayCntr_H			res 1
UpdateDelayCntr				res 1
RotationCntr				res 1

#define UpdateDelayCnt		0xff
#define MinPWM				0x00	;0x017	;0x03A
#define MaxPWM				0x01E8
;-------------------------------------------------------------------------------
; Counter used in the cycle/cycle peak current limit using the /FLTA input
;-------------------------------------------------------------------------------
#define	MaxFaultAcnt		0xEF
PWMcycleCntr				res	1
FaultAcntr					res	1

;-------------------------------------------------------------------------------
; Control registers bit definitions
; see the end of the HIP4086 Motor Drive.asm file for complete bit definitions
; for the registers used in this application
;-------------------------------------------------------------------------------
#define	ADCbufOverFlow_b	ADCON1,BFOVFL
#define ADCenConvert_b		ADCON0,GO
#define ADCintFlag_b		PIR1,ADIF
#define ADCintEnable_b		PIE1,ADIE
#define	HallAintEnable_b	PIE3,IC1IE
#define HallAintFlag_b		PIR3,IC1IF
#define	HallBintEnable_b	PIE3,IC2QEIE
#define HallBintFlag_b		PIR3,IC2QEIF
#define HallCintEnable_b	PIE3,IC3DRIE
#define HallCintFlag_b		PIR3,IC3DRIF
#define	PWMintEnable_b		PIE3,PTIE
#define	PWMintFlag_b		PIR3,PTIF
#define PeripheralIntEn_b	INTCON,PEIE	
#define	GlobalIntEnable_b	INTCON,GIE
#define PWMenable_b			PTCON1,PTEN
#define	Timer0IntEnable_b	INTCON,TMR0IE
#define Timer0IntFlag_b		INTCON,TMR0IF
#define	Timer1IntEnable_b	PIE1,TMR1IE
#define	Timer1IntFlag_b		PIR1,TMR1IF
;-------------------------------------------------------------------------------
; FLAG DEFINITIONS  
;-------------------------------------------------------------------------------
UpdatePWM_B					res	1		;note that an entire byte is used for a
OverCurFault_B				res	1		;flag bit (easier for trouble-shooting)
StopMotor_B					res	1
ForwardMotor_B				res	1
SeqEnabled_B				res 1
TestModeActive_B			res 1
EnableBlink_B				res 1
EnableLEDs_B				res 1
ToggleBlink_B				res 1
MotorRunning_B				res 1
ReverseEnabled_B			res 1

#define ReverseEnabled_b	ReverseEnabled_B,0
#define MotorRunning_b		MotorRunning_B,0
#define UpdatePWM_b			UpdatePWM_B,0
#define	OverCurFault_b		OverCurFault_B,0	
#define	StopMotor_b			StopMotor_B,0	
#define	ForwardMotor_b		ForwardMotor_B,0	
#define	SeqEnabled_b		SeqEnabled_B,0
#define TestModeActive_b	TestModeActive_B,0
#define ToggleBlink_b		ToggleBlink_B,0
#define	LED0Enabled_b		EnableLEDs_B,4
#define LED1Enabled_b		EnableLEDs_B,5
#define LED2Enabled_b		EnableLEDs_B,6
#define LED3Enabled_b		EnableLEDs_B,7

;------------------------------------------------------------------------------
;PWM frequency definition. PTPER register sets up the PWM Period
;------------------------------------------------------------------------------
#define OSCILLATOR				d'40000000' ;10Mhz crystal x 4 (HPLL mode)
#define PWM_TIMER_PRESCALE		d'01'
#define	PWM_FREQUENCY			d'20000'
PTPER_VALUE = (OSCILLATOR/(4*PWM_FREQUENCY*PWM_TIMER_PRESCALE))-1

;------------------------------------------------------------------------------
; Configuration bits
;
; These definitions set up various hardware features of the PIC18F2431.
;------------------------------------------------------------------------------
	LIST p=18f2431,f=INHX32

 CONFIG OSC = HSPLL		;HS oscillator, PLL enabled (clock frequency = 4 x FOSC1)
 CONFIG FCMEN = OFF		;Fail-Safe Clock Monitor Enable bit
 CONFIG IESO = OFF		;Internal External Oscillator Switchover bit
 CONFIG PWRTEN = ON		;Power-up Timer Enable bit
 CONFIG BOREN = OFF		;Brown-out Reset Enable bits
 CONFIG BORV = 27		;Brown Out Reset Voltage bits
 CONFIG WDTEN = OFF		;Watchdog Timer Enable bit
 CONFIG WDPS = 32768	;Watchdog Timer Enable bit
 CONFIG WINEN = OFF		;Watchdog Timer Window Enable bit
 CONFIG PWMPIN = OFF	;PWM outputs disabled upon Reset (default)
 CONFIG LPOL = HIGH		;PWM0, 2, 4 and 6 are active-high
 CONFIG HPOL = LOW		;PWM1, 3, 5 and 7 are active-low
 CONFIG T1OSCMX = ON	;Low-power Timer1 operation when microcontroller is in Sleep mode 
 CONFIG MCLRE = ON		;MCLR i/o enable bit
 ;CONFIG EXCLK = OFF		;external clock input muxed with RD0
 ;CONFIG PWM4MX = ON		;PWM4 muxed with RB5
 ;CONFIG	SSPMX = OFF		;ssp muxed with PORTD
 ;CONFIG	FLTAMX = ON		;/FLTA INPUT MUXED WITH RC1
 CONFIG STVREN = ON		;Stack full/underflow will cause Reset 
 CONFIG LVP = OFF		;Low-voltage ICSP disabled
 CONFIG DEBUG = ON		;Background debugger enabled,RB6 and RB7 are dedicated to In-Circuit Debug
 CONFIG CP0 = OFF		;Block 0 (000200-000FFFh) not code-protected
 CONFIG CP1 = OFF		;Block 1 (001000-001FFF) not code-protected
 CONFIG CP2 = OFF		;Block 2 (002000-002FFFh) not code-protected
 CONFIG CP3 = OFF		;Block 3 (003000-003FFFh) code-protected
 CONFIG CPB = OFF		;Boot Block (000000-0001FFh) not code-protected
 CONFIG CPD = OFF		;Data EEPROM not code-protected
 CONFIG WRT0 = OFF		;Block 0 (000200-000FFFh) not write-protected
 CONFIG WRT1 = OFF		;Block 1 (001000-001FFF) not write-protected
 CONFIG WRT2 = OFF		;Block 2 (002000-002FFFh) not write-protected
 CONFIG WRT3 = OFF		;Block 3 (003000-003FFFh) not write-protected
 CONFIG WRTC = OFF		;Configuration registers (300000-3000FFh) not write-protected
 CONFIG WRTB = OFF		;Boot Block (000000-0001FFh) not write-protected
 CONFIG WRTD = OFF		;Data EEPROM not write-protected
 CONFIG EBTR0 = OFF		;Block 0 (000200-000FFFh) not protected from table reads executed in other blocks
 CONFIG EBTR1 = OFF		;Block 1 (001000-001FFF) not protected from table reads executed in other blocks
 CONFIG EBTR2 = OFF		;Block 2 (002000-002FFFh) not protected from table reads executed in other blocks
 CONFIG EBTR3 = OFF		;Block 3 (003000-003FFFh) not protected from table reads executed in other blocks
 CONFIG EBTRB = OFF		;Boot Block (000000-0001FFh) not protected from table reads executed in other blocks
 
;-------------------------------------------------------------------------------
;    Example of a commutation sequence from typical winding energizing table
;-------------------------------------------------------------------------------
		; |	Hall A | Hall B | Hall C | Phase A | Phase B | Phase C |
		;-----------------------------------------------------------
		; |	   0   |	0	|	0	 |	---	   |	---	 |	---	   |
		; |	   1   |	0	|	0	 |	DC+	   |	---	 |	DC-	   |
		; |	   0   |	1	|	0	 |	---	   |	DC-	 |	DC+    |
		; |	   1   |	1	|	0	 |	DC+	   |	DC-	 |	---    |
		; |	   0   |	0	|	1	 |	DC-	   |	DC+	 |	---    |
		; |	   1   |	0	|	1	 |	---	   |	DC+	 |	DC-    |
		; |	   0   |	1	|	1	 |	DC-	   |    ---	 |	DC+    |
		; |	   1   |	1	|	1	 |	---	   |	---	 |	---    |
		;-----------------------------------------------------------


